/**
 * \file: exchnd_collector_ptrace.c
 *
 * Ptrace specific collector implementation
 * Some data collected in the exception handler is collected in the user
 * space. The file handle ptrace related arch generic gathering.
 *
 * \component: Exception handler
 *
 * \author: Frederic Berat <fberat@de.adit-jv.com>
 *
 * \copyright (c) 2010, 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <ctype.h>
#include <dlfcn.h>
#include <limits.h>
#include <signal.h>
#include <stddef.h>
#include <stdbool.h>
#include <libunwind-ptrace.h>

#include <sys/ioctl.h>
#include <sys/ptrace.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <linux/exchnd.h>

#include "exchnd_collector.h"

#ifndef min
#   define min(a, b) (((a) < (b)) ? (a) : (b))
#endif

#ifndef max
#   define max(a, b) (((a) > (b)) ? (a) : (b))
#endif

static const unsigned int ULSZ = sizeof(unsigned long);

#ifndef PTRACE_SEIZE
/* Until new eglibc is provided from intel, these macro definitions
 * are coded in this file.
 */
/** Request **/
#   define PTRACE_SEIZE        0x4206
#endif
#ifndef PTRACE_INTERRUPT
#   define PTRACE_INTERRUPT    0x4207
#endif
#ifndef PTRACE_LISTEN
#   define PTRACE_LISTEN       0x4208
#endif
/** Options **/
#ifndef PTRACE_O_EXITKILL
#   define PTRACE_O_EXITKILL   (1 << 20)
#   ifdef PTRACE_O_MASK
#      undef PTRACE_O_MASK
#   endif /* PTRACE_O_MASK */
#   define PTRACE_O_MASK       (0x000000ff | PTRACE_O_EXITKILL)
#endif /* PTRACE_O_EXITKILL */

extern long attached[MAX_THREADS];
extern int idx_max;

/* Libunwind functions */
int (*unw_get_reg_p)(unw_cursor_t *, unw_regnum_t, unw_word_t *);
static int (*unw_get_fpreg_p)(unw_cursor_t *, unw_regnum_t, unw_fpreg_t *);
static int (*unw_get_saveloc_p)(unw_cursor_t *, unw_regnum_t,
                                unw_save_loc_t *);
static int (*unw_is_signal_frame_p)(unw_cursor_t *);
static int (*unw_step_p)(unw_cursor_t *);
static int (*unw_init_remote_p)(unw_cursor_t *, unw_addr_space_t, void *);
extern int (*unw_getcontext_p)(unw_context_t *);
static int (*unw_init_local_p)(unw_cursor_t *, unw_context_t *);
static unw_addr_space_t (*unw_create_addr_space_p)(unw_accessors_t *, int);
static void (*unw_destroy_addr_space_p)(unw_addr_space_t);
static unw_accessors_t *(*unw_get_accessors_p)(unw_addr_space_t);
static int (*unw_get_proc_name_p)(unw_cursor_t *, char *, int, unw_word_t *);

static void *(*_UPT_create_p)(int);
static int (*_UPT_destroy_p)(void *);

static char *get_reg_name = STRINGIFY(UNW_OBJ(get_reg));
static char *get_fpreg_name = STRINGIFY(UNW_OBJ(get_fpreg));
static char *get_saveloc_name = STRINGIFY(UNW_OBJ(get_save_loc));
static char *is_signal_frame_name = STRINGIFY(UNW_OBJ(is_signal_frame));
static char *step_name = STRINGIFY(UNW_OBJ(step));
static char *init_remote_name = STRINGIFY(UNW_OBJ(init_remote));
static char *init_local_name = STRINGIFY(UNW_OBJ(init_local));
static char *local_addr_space_name = STRINGIFY(UNW_OBJ(local_addr_space));
static char *create_addr_space_name = STRINGIFY(UNW_OBJ(create_addr_space));
static char *destroy_addr_space_name = STRINGIFY(UNW_OBJ(destroy_addr_space));
static char *get_accessors_name = STRINGIFY(UNW_OBJ(get_accessors));
static char *get_proc_name_name = STRINGIFY(UNW_OBJ(get_proc_name));
static unw_accessors_t *_UPT_accessors_p;
static unw_addr_space_t *unw_local_addr_space_p;

#if 0 /*def unw_set_unwind_method */
static int (*unw_set_unwind_method_p)(int);
static char *set_unwind_method_name = STRINGIFY(UNW_OBJ(set_unwind_method));
#endif

void *libunwind_handle;
void *libunwind_ptrace_handle;

/*
 * \func libunwind_load
 *
 * Load libunwind libraries and get needed functions symbols.
 *
 */
int libunwind_load(void)
{
    libunwind_handle = dlopen(libunwind_arch_lib(), RTLD_NOW | RTLD_GLOBAL);

    if (libunwind_handle == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        return 0;
    }

    libunwind_ptrace_handle = dlopen("libunwind-ptrace.so.0",
                                     RTLD_NOW | RTLD_GLOBAL);

    if (libunwind_handle == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        dlclose(libunwind_handle);
        return 0;
    }

    /* Initialize pointers to the dynamic library functions we will use.  */

    unw_get_reg_p = dlsym(libunwind_handle, get_reg_name);

    if (unw_get_reg_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_get_fpreg_p = dlsym(libunwind_handle, get_fpreg_name);

    if (unw_get_fpreg_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_get_saveloc_p = dlsym(libunwind_handle, get_saveloc_name);

    if (unw_get_saveloc_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_is_signal_frame_p = dlsym(libunwind_handle, is_signal_frame_name);

    if (unw_is_signal_frame_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_step_p = dlsym(libunwind_handle, step_name);

    if (unw_step_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_init_local_p = dlsym(libunwind_handle, init_local_name);

    if (unw_init_local_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_init_remote_p = dlsym(libunwind_handle, init_remote_name);

    if (unw_init_remote_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_local_addr_space_p = dlsym(libunwind_handle, local_addr_space_name);

    if (unw_local_addr_space_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_create_addr_space_p = dlsym(libunwind_handle, create_addr_space_name);

    if (unw_create_addr_space_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_destroy_addr_space_p = dlsym(libunwind_handle, destroy_addr_space_name);

    if (unw_destroy_addr_space_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    unw_get_accessors_p = dlsym(libunwind_handle, get_accessors_name);

    if (unw_get_accessors_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

#if 0 /*def unw_set_unwind_method */
    unw_set_unwind_method_p = dlsym(libunwind_handle, set_unwind_method_name);

    if (unw_set_unwind_method_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

#endif

    unw_get_proc_name_p = dlsym(libunwind_handle, get_proc_name_name);

    if (unw_get_proc_name_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    _UPT_create_p = dlsym(libunwind_ptrace_handle, "_UPT_create");

    if (_UPT_create_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    _UPT_destroy_p = dlsym(libunwind_ptrace_handle, "_UPT_destroy");

    if (_UPT_destroy_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    _UPT_accessors_p = dlsym(libunwind_ptrace_handle, "_UPT_accessors");

    if (_UPT_accessors_p == NULL) {
        exchnd_print_error("Failed to load libunwind: %s\n", dlerror());
        libunwind_unload();
        return 0;
    }

    return libunwind_load_arch(libunwind_handle);
}

/*
 * \func libunwind_unload
 *
 * Unload trace libraries
 */
void libunwind_unload(void)
{
    libunwind_unload_arch();
    dlclose(libunwind_handle);
    dlclose(libunwind_ptrace_handle);
}

void show_backtrace(void)
{
    unw_cursor_t cursor;
    unw_context_t uc;
    unw_word_t ip;
    char proc_name[256];
    unw_word_t offset;

    unw_getcontext_p(&uc);

    unw_init_local_p(&cursor, &uc);
    exchnd_print_error("Backtrace:");

    while (unw_step_p(&cursor) > 0) {
        unw_get_reg_p(&cursor, UNW_REG_IP, &ip);

        if (unw_get_proc_name_p(&cursor, proc_name, sizeof(proc_name),
                                &offset) < 0)
            strcpy(proc_name, "");

        exchnd_print_error(TMPL_BT, proc_name, offset, ip);
    }
}

/* Cleanup libunwind structures */
void libunwind_clean(struct exchnd_unwind *eunw)
{
    if (!eunw)
        return;

    if (eunw->upt_info)
        _UPT_destroy_p(eunw->upt_info);

    if (eunw->addr_space)
        unw_destroy_addr_space_p(eunw->addr_space);

    /* The maps have to be freed separately as they can be used
     * multiple times.
     */
    memset(eunw, 0, offsetof(struct exchnd_unwind, maps));
}

/* Prepare libunwind structures */
int libunwind_prepare(pid_t pid, struct exchnd_unwind *eunw)
{
    int result;
    int retries = MAX_RETRIES;

    eunw->addr_space = unw_create_addr_space_p(_UPT_accessors_p, 0);

    if (!(eunw->addr_space)) {
        exchnd_log_error("Failed to initialize unwind address space");
        return -1;
    }

    eunw->upt_info = _UPT_create_p(pid);

    if (eunw->upt_info == NULL) {
        exchnd_log_error("Failed to create UPT info header");
        unw_destroy_addr_space_p(eunw->addr_space);
        return -1;
    }

    result = unw_init_remote_p(&eunw->cursor, eunw->addr_space, eunw->upt_info);

    while ((result < 0) && (retries-- > 0)) {
        usleep(TEN_MSECS);
        result = unw_init_remote_p(&eunw->cursor,
                                   eunw->addr_space,
                                   eunw->upt_info);
    }

    if (result < 0) {
        exchnd_log_error("Failed to initialize unwind cursor (%d)", result);
        libunwind_clean(eunw);
        return -1;
    }

    eunw->accessors = unw_get_accessors_p(eunw->addr_space);

    return 0;
}

static int interrupt_process(pid_t pid)
{
    int ret = 0;
    int trial = MAX_RETRIES;

    ret = ptrace(PTRACE_INTERRUPT, pid, 0, 0);

    while ((ret < 0) && trial--) {
        if (trial == MAX_RETRIES / 2)
            exchnd_print_error("Failed to INTERRUPT thread %d (%s).",
                               pid,
                               strerror(errno));

        usleep(TEN_MSECS);
        ret = ptrace(PTRACE_INTERRUPT, pid, 0, 0);
    }

    ret = ptrace(PTRACE_LISTEN, pid, 0, 0);
    trial = MAX_RETRIES;

    while ((ret < 0) && trial--) {
        if (trial == MAX_RETRIES / 2)
            exchnd_print_error("Failed to LISTEN thread %d (%s).",
                               pid,
                               strerror(errno));

        usleep(TEN_MSECS);
        ret = ptrace(PTRACE_LISTEN, pid, 0, 0);
    }

    ret = ptrace(PTRACE_INTERRUPT, pid, 0, 0);
    trial = MAX_RETRIES;

    while ((ret < 0) && trial--) {
        if (trial == MAX_RETRIES / 2)
            exchnd_print_error("Failed to INTERRUPT thread %d (%s).",
                               pid,
                               strerror(errno));

        usleep(TEN_MSECS);
        ret = ptrace(PTRACE_INTERRUPT, pid, 0, 0);
    }

    return ret;
}

int pt_attach(pid_t pid, int exitkill)
{
    long pattach;
    long options = PTRACE_O_MASK;

    if (!exitkill)
        options &= ~PTRACE_O_EXITKILL;

    exchnd_ping_driver();
    pattach = ptrace(PTRACE_SEIZE, pid, 0, options);

    if (pattach < 0) {
        exchnd_log_error("Attach on %d failed with error %s\n",
                         pid,
                         strerror(errno));
        return -errno;
    }

    interrupt_process(pid);

    return 0;
}

int pt_group_attach(ExchndInterface_t *exh_if,
                    struct exchnd_message_header *header,
                    int pid)
{
    int attach_failed = 1;
    int mustkill = (action & EXCHND_KILL);

    if ((pid == header->pid) &&
        is_tracee_attached(exh_if->PTRACE_attached)) {
        /* No need to attach again */
        exh_if->PTRACE_attached = EXH_GROUP;
        attached[idx_max++] = pid;
        return idx_max;
    }

    attach_failed = pt_attach(pid, mustkill);

    if (attach_failed != 0) {
        exchnd_log_error("Attach on pid %d failed.", pid);
        return idx_max;
    }

    attached[idx_max++] = pid;

    if (!is_tracee_attached(exh_if->PTRACE_attached))
        exh_if->PTRACE_attached = EXH_PARTIAL_GROUP;
    else
        exh_if->PTRACE_attached = EXH_GROUP;

    return idx_max;
}

int pt_detach(int pid, int att_level)
{
    long thread_pid = 0;
    int rc = 0;
    int failure = 0;

    int idx = 0;

    if (att_level == EXH_DETACHED)
        return rc;

    if (att_level == EXH_THREAD)
        return ptrace(PTRACE_DETACH, pid, 0, 0);

    /* att_level == EXH_GROUP */

    while (idx < idx_max) {
        thread_pid = attached[idx];

        if (ptrace(PTRACE_DETACH, thread_pid, 0, 0)) {
            /* Continue detaching even on failure.
             * We shall try to detach every threads.
             * Keep last failure for return code.
             */
            failure = -errno;

            /* If ESRCH the thread is probably already dead.
             * Just ignore it as there is nothing we can do.
             * Detach will be forced by the end of the tracing
             * thread.
             */
            if (errno != ESRCH)
                exchnd_log_error("%s failed for pid %d (%s)",
                                 __func__,
                                 thread_pid,
                                 strerror(errno));
        }

        idx++;
    }

    if (failure)
        rc = failure;

    return rc;
}

/*
 * \func __get_aligned_mem
 *
 * Collect the memory using the libunwind interface, and potentially makes
 * a copy of it in a dedicated output.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the line and the phrase buffer.
 */
static unw_word_t __get_aligned_mem(struct exchnd_unwind *eunw,
                                    struct dump_mem *dm,
                                    unsigned int to_print)
{
    unw_word_t i = dm->add;
    unsigned int j = 0;

    for (; i < dm->add + to_print * ULSZ; i += ULSZ, j++) {
        if (i < dm->start)
            continue;

        eunw->accessors->access_mem(eunw->addr_space,
                                    i,
                                    &dm->line[j],
                                    0,
                                    eunw->upt_info);

        if (dm->out) {
            *dm->out = dm->line[j];
            dm->out++;
        }
    }

    return i;
}

/*
 * \func __build_phrase
 *
 * Store the character translation of a line in a phrase buffer.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the line and the phrase buffer.
 */
static int __build_phrase(struct dump_mem *dm, unsigned int to_print)
{
    unsigned int j = 0;
    unsigned int s = 0;
    char *c = (char *)dm->line;
    int zeroed = 1;
    unsigned int align = (dm->start > dm->add) ? dm->start - dm->add : 0;

    /* Building the phrase */
    for (j = 0; j < to_print * ULSZ; j++) {
        if (j < align) {
            dm->phrase[j + s] = ' ';
            continue;
        }

        if (j && (j % (ULSZ * 4) == 0)) {
            dm->phrase[j + s] = ' ';
            s++;
        }

        if (*(c + j) != 0)
            zeroed = 0;

        if (isprint(*(c + j)))
            dm->phrase[j + s] = *(c + j);
        else
            dm->phrase[j + s] = '.';
    }

    return zeroed;
}

/*
 * \func __print_phrase
 *
 * Print the line and phrase buffers in the storages.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the line and the phrase buffer.
 */
static void __print_phrase(struct dump_mem *dm, unsigned int to_print)
{
    unsigned int j = 0;
    char buffer[MAX_MSG_LEN] = "";
    int len = 0;
    unsigned int align = (dm->start > dm->add) ? dm->start - dm->add : 0;

    len += snprintf(buffer, sizeof(buffer), "<"TMPL_MEM ">", dm->add);

    for (j = 0; j < dm->w; j++) {
        if (j < align / ULSZ) {
            len += snprintf(buffer + len,
                            sizeof(buffer) - len,
                            TMPL_BLANK);
            continue;
        }

        if (j && (j % ULSZ == 0))
            len += snprintf(buffer + len,
                            sizeof(buffer) - len,
                            "  ");

        if (j < to_print)
            len += snprintf(buffer + len,
                            sizeof(buffer) - len,
                            " "TMPL_MEM,
                            dm->line[j]);
        else
            len += snprintf(buffer + len,
                            sizeof(buffer) - len,
                            TMPL_BLANK);
    }

    len += snprintf(buffer + len,
                    sizeof(buffer) - len,
                    " %s\n",
                    dm->phrase);

    exchnd_backend_store(buffer, strlen(buffer));
}

/*
 * \func __dump_mem
 *
 * Dump the memory based on input command.
 * May skip 0 field if they are duplicate line by line.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the dump command information
 *
 * Example dm->l=2, dm->w=3, dm->end = dm->start + 16, dm->start = 0x1234,
 * dm->add = 0x1230:
 * <(1230)>           bbbbbbbb bbbbbbbb     ........
 * <(123c)>  bbbbbbbb bbbbbbbb          ........
 */
static void __dump_mem(struct exchnd_unwind *eunw, struct dump_mem *dm)
{
    int skip = 0;

    while ((dm->add < dm->end) &&
           (dm->add < dm->start + dm->l * dm->w * ULSZ)) {
        int zeroed = 0;
        unsigned int to_print = min((unsigned int)((dm->end - dm->add) / ULSZ),
                                    dm->w);
        unw_word_t tmp = 0x0UL;

        memset(dm->phrase, '\0', (dm->w + 1) * ULSZ);
        memset(dm->line, 0, dm->w * ULSZ);

        tmp = __get_aligned_mem(eunw, dm, to_print);

        zeroed = __build_phrase(dm, to_print);

        if (zeroed) {
            if (skip++) {
                /* Increase the maximum amount of line and
                 * skip the current */
                dm->l += 1;

                if (skip == 2)
                    exchnd_backend_store("--- Skip zeroed",
                                         strlen("--- Skip zeroed"));

                if (skip > 100)
                    break;

                dm->add = tmp;
                continue;
            }
        } else {
            skip = 0;
        }

        __print_phrase(dm, to_print);
        dm->add = tmp;
    }
}

/*
 * \func exchnd_dump_aligned_mem
 *
 * Helper function that dumps an area of memory.
 * May skip 0 field if they are duplicate line by line.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the dump command information
 *
 * Example dm->l=2, dm->w=3, dm->end = dm->start + 16, dm->start = 0x1234:
 * <(1230)>           bbbbbbbb bbbbbbbb     ........
 * <(123c)>  bbbbbbbb bbbbbbbb          ........
 */
static void exchnd_dump_aligned_mem(struct exchnd_unwind *eunw,
                                    struct dump_mem *dm)
{
    unw_word_t line[dm->w];
    char phrase[(dm->w + 1) * ULSZ];

    dm->add = dm->start & ~0xfUL;
    dm->line = line;
    dm->phrase = phrase;

    __dump_mem(eunw, dm);
}

/*
 * \func exchnd_dump_mem
 *
 * Helper function that dumps an area of memory.
 * May skip 0 field if they are duplicate line by line.
 *
 * \param eunw Structure containing unwinding data.
 * \param dm   Structure containing the dump command information
 *
 * Example dm->l=2, dm->w=3, dm->end = dm->start + 20:
 * <(start+0)>  bbbbbbbb bbbbbbbb bbbbbbbb ............
 * <(start+c)>  bbbbbbbb bbbbbbbb          ........
 */
void exchnd_dump_mem(struct exchnd_unwind *eunw,
                     struct dump_mem *dm)
{
    unw_word_t line[dm->w];
    char phrase[(dm->w + 1) * ULSZ];

    dm->add = dm->start;
    dm->line = line;
    dm->phrase = phrase;

    __dump_mem(eunw, dm);
}

/*
 * \func get_maps
 *
 * This helper function gets the map associated to an address
 *
 * \param reg_pc Address for which the path name shall be found.
 * \param eunw Structure containing unwinding and maps data.
 *
 * \return The found map if successful. NULL otherwise.
 */
static struct exchnd_maps *get_map(unsigned long reg_pc,
                                   struct exchnd_unwind *eunw)
{
    int i = 0;
    struct exchnd_maps *rc = NULL;

    if (!eunw)
        return rc;

    for (i = 0; i < eunw->num_maps; i++) {
        if (eunw->maps[i].start > reg_pc)
            break;

        if ((eunw->maps[i].start <= reg_pc) && (reg_pc <= eunw->maps[i].end)) {
            rc = &eunw->maps[i];
            break;
        }
    }

    return rc;
}

#define NUM_REG 64 /* White lie */

static void exchnd_get_arch_regs(struct exchnd_unwind *eunw,
                                 unw_word_t(*dump)[NUM_REG],
                                 unsigned int *to_dump)
{
    unsigned int i;
    unsigned int first_reg = get_arch_first_reg();
    unsigned int last_reg = get_arch_last_reg();

    /* Now get and check registers one by one */
    for (i = first_reg; i <= last_reg; i++) {
        int found = 0;
        unsigned int j;

        unw_get_reg_p(&eunw->cursor, i, &((*dump)[*to_dump]));

        for (j = 0; j < *to_dump; j++)
            if ((*dump)[*to_dump] == (*dump)[j])
                found = 1;

        if (found)
            continue;

        if (get_map((*dump)[*to_dump], eunw))
            (*to_dump)++;
    }
}

/*
 * \func exchnd_indirect_regs_dump
 *
 * This function dumps memory potentially pointed to by addresses read
 * from registers.
 *
 * \param eunw The exchnd unwind structure
 */
void exchnd_indirect_regs_dump(struct exchnd_unwind *eunw)
{
    char buffer[MAX_MSG_LEN];

    unw_word_t dump[NUM_REG] = { 0 };
    unsigned int dump_id = 0;
    unsigned int max_dump = 0;

    exchnd_get_arch_regs(eunw, &dump, &max_dump);

    while (dump_id < max_dump) {
        struct dump_mem dm = {
            .start = dump[dump_id] - 16 * ULSZ,
            .end = dump[dump_id] + 32 * ULSZ,
            /* Line to dump should be even */
            .l = 8,
            .w = 8
        };

        snprintf(buffer,
                 MAX_MSG_LEN,
                 "Dump around 0x"TMPL_MEM " (0x"TMPL_MEM " - 0x"TMPL_MEM ")",
                 dump[dump_id],
                 dm.start,
                 dm.end);
        exchnd_backend_store(buffer, strlen(buffer));
        exchnd_dump_aligned_mem(eunw, &dm);

        dump_id++;
    }
}

/* May need to be platform specific */
static char *func_list[] = {
    "_start",
    "__libc_start_main",
    "clone",
    NULL
};

static char *white_map[] = {
    "[vdso]",
    "[vsyscall]",
    "[sigpage]",
    "[vectors]",
    NULL
};

static bool check_string(char **list, char *string)
{
    int i = 0;

    while (list[i] != NULL) {
        if (strncmp(list[i], string, strlen(list[i])) == 0)
            return true;

        i++;
    }

    return false;
}

static int exchnd_validate_pc(struct exchnd_unwind *eunw,
                              unw_word_t pc,
                              char *func)
{
    int ret = EXCHND_FB_FRAME;
    struct exchnd_maps *map = get_map(pc, eunw);

    /* Return if we are in vdso */
    if ((func[0] == '\0') &&
        map && map->exec &&
        ((map == eunw->maps) || check_string(white_map, map->name)))
        return EXCHND_FB_NONE;

    if (!strlen(func))
        return EXCHND_FB_FRAME;

    if (check_string(func_list, func))
        ret = EXCHND_FB_NONE;

    return ret;
}

/*\func exchnd_get_func_at_address
 *
 * Find the function at given address and store the result in the given buffer.
 * The buffer to be used for dumping needs to be provided.
 *
 * \param pc The address to resolve.
 * \param eunw Structure containing unwinding and maps data.
 * \param buf_cursor The buffer were to put the function name to
 * \param buf_size The size of the buffer
 * \param last_func The previous function name buffer for later comparison
 */
int exchnd_get_func_at_address(unw_word_t pc,
                               struct exchnd_unwind *eunw,
                               char *buf_cursor,
                               int buf_size,
                               char *last_func)
{
    unw_word_t offset;
    int char_count = 0;
    int ret = 0;
    char proc_name[256] = { '\0' };
    struct exchnd_maps *map = NULL;

    if (!buf_cursor)
        return 0;

    map = get_map(pc, eunw);

    if (map) {
        char_count = snprintf(buf_cursor, buf_size, "%s", map->name);
        buf_cursor += char_count;
        buf_size -= char_count;
    } else {
        exchnd_log_error("No map corresponding to %p found", (void *)pc);
        return -UNW_EINVALIDIP;
    }

    if (last_func)
        memset(last_func, '\0', MAX_MSG_LEN);

    ret = eunw->accessors->get_proc_name(eunw->addr_space, pc,
                                         proc_name, sizeof(proc_name),
                                         &offset, eunw->upt_info);

    if (ret >= 0) {
        char_count = snprintf(buf_cursor, buf_size,
                              TMPL_BT_1, proc_name, offset);

        if (last_func)
            snprintf(last_func, MAX_MSG_LEN, "%s", proc_name);

        buf_cursor += char_count;
        buf_size -= char_count;

        ret = 0;
    }

    snprintf(buf_cursor, buf_size, TMPL_BT_2, pc);

    return ret;
}

void exchnd_extract_vars(struct exchnd_unwind *eunw,
                         unw_word_t *dump_in,
                         unsigned int tsize,
                         unw_word_t start,
                         unw_word_t stop)
{
    unsigned int found = 0;
    unsigned int to_dump = 0;

    struct dump_mem dm = {
        /* Line to dump should be even */
        .l = (tsize / 8) + 1,
        .w = 8,
        .out = dump_in
    };

    if ((tsize <= 0) || ((stop - start) / ULSZ > tsize))
        return;

    if (start < stop) {
        dm.start = start;
        dm.end = stop;
        exchnd_dump_aligned_mem(eunw, &dm);
    }

    while (found < tsize) {
        if (get_map(dump_in[found], eunw))
            dump_in[to_dump++] = dump_in[found];

        found++;
    }

    while (to_dump < tsize)
        dump_in[to_dump++] = 0UL;
}

void exchnd_indirect_dump(struct exchnd_unwind *eunw,
                          unw_word_t *to_dump,
                          unsigned int tsize)
{
    unsigned int inc = 0;
    char buffer[SHRT_MSG_LEN] = { 0 };

    while ((inc < tsize) && to_dump[inc]) {
        struct exchnd_maps *map = get_map(to_dump[inc], eunw);
        struct dump_mem dm = {
            .start = to_dump[inc] - 16 * ULSZ,
            .end = to_dump[inc] + 32 * ULSZ,
            /* Line to dump should be even */
            .l = 8,
            .w = 8
        };

        if (!map)
            continue;

        snprintf(buffer, SHRT_MSG_LEN,
                 "Dump around 0x"TMPL_MEM
                 " (0x"TMPL_MEM " - 0x"TMPL_MEM ") in %s",
                 to_dump[inc],
                 dm.start,
                 dm.end,
                 map->name);
        exchnd_backend_store(buffer, strlen(buffer));

        exchnd_dump_aligned_mem(eunw, &dm);

        inc++;
    }
}

/*
 * \func exchnd_extend_bt
 *
 * Generic function to be used by exchnd_unwind to extend data provided while
 * unwinding.
 *
 * \param eunw Structure containing unwinding and maps data.
 * \param prev_sp Previous SP value to search for arguments
 */
void exchnd_extend_bt(struct exchnd_unwind *eunw, unw_word_t *prev_sp)
{
    char buffer[MAX_MSG_LEN];
    unw_word_t reg_sp = 0x0UL;
    unw_word_t org_sp = 0x0UL;
    unw_word_t reg_lr = 0x0UL;
    unw_word_t reg_fp = 0x0UL;
    unw_word_t start_sp = 0x0UL;

    unw_word_t dump[ARG_CNT * 2] = { 0UL };

    struct exchnd_maps *sp_map = NULL;
    unsigned long map_end = ULONG_MAX;
    unsigned long map_start = 0;

    unw_word_t end_sp = 0UL;

    if (unw_get_reg_p(&eunw->cursor, get_arch_lr_reg(), &reg_lr) < 0)
        reg_lr = ULONG_MAX;

    if (unw_get_reg_p(&eunw->cursor, get_arch_sp_reg(), &reg_sp) < 0)
        reg_sp = ULONG_MAX;

    if (unw_get_reg_p(&eunw->cursor, get_arch_fp_reg(), &reg_fp) < 0)
        reg_fp = ULONG_MAX;

    if (eunw->num_maps) {
        /* Properly initialize map_start and map_end in any case */
        map_start = eunw->maps[0].start;
        map_end = eunw->maps[eunw->num_maps - 1].end;
    }

    if (reg_sp == *prev_sp) {
        sp_map = get_map(reg_sp, eunw);

        if (!sp_map) {
            exchnd_log_error("Stack map not found !");
            return;
        }

        map_end = sp_map->end;
        map_start = sp_map->start;

        reg_sp = map_end;
        reg_lr = 0UL;
    }

    org_sp = reg_sp;

    snprintf(buffer, MAX_MSG_LEN, EXT_REG_TMPL, reg_lr, reg_fp, reg_sp);
    exchnd_backend_store(buffer, strlen(buffer));

    if (reg_sp == ULONG_MAX) {
        exchnd_log_error("---");
        return;
    }

    /* SP points on the first argument for current PC call. */
    reg_sp -= ULSZ;

    if (*prev_sp && (*prev_sp < reg_sp)) {
        start_sp = max(*prev_sp, (reg_sp - ARG_CNT * ULSZ));

        snprintf(buffer, MAX_MSG_LEN,
                 "Saved registers and local vars (%#lx - %#lx):\n",
                 (unsigned long)start_sp,
                 (unsigned long)reg_sp - ULSZ);
        exchnd_backend_store(buffer, strlen(buffer));

        exchnd_extract_vars(eunw, dump, ARG_CNT, start_sp, reg_sp);
    }

    reg_sp = org_sp;

    end_sp = min((reg_sp + ARG_CNT * ULSZ), map_end);

    if ((end_sp > reg_sp) && (map_start < reg_sp)) {
        snprintf(buffer, MAX_MSG_LEN,
                 "Potential args (%#lx - %#lx):\n",
                 (unsigned long)reg_sp,
                 (unsigned long)end_sp - ULSZ);
        exchnd_backend_store(buffer, strlen(buffer));
        exchnd_extract_vars(eunw, &dump[ARG_CNT], ARG_CNT, reg_sp, end_sp);
    }

    exchnd_indirect_dump(eunw, dump, ARG_CNT);
    exchnd_indirect_dump(eunw, &dump[ARG_CNT], ARG_CNT);

    snprintf(buffer, MAX_MSG_LEN, "---");
    exchnd_backend_store(buffer, strlen(buffer));

    *prev_sp = org_sp;
}

/*
 * \func exchnd_unwind
 *
 * Generic function to be used by log_backtrace to unwind frames.
 *
 * \param eunw Structure containing unwinding and maps data.
 * \param prev_pc Previous PC value to check recursion
 * \param prev_sp Previous SP value to search for arguments
 * \param extend Shall the backtrace be extended to provide arguments and
 *        memory dump.
 * \param last_func The previous function name buffer for later comparison
 */
int exchnd_unwind(struct exchnd_unwind *eunw,
                  unw_word_t *prev_pc,
                  unw_word_t *prev_sp,
                  int extend,
                  char *last_func)
{
    unw_word_t reg_pc;

    char buffer[MAX_MSG_LEN];
    char *buf_cursor = buffer;
    int buf_size = sizeof(buffer);
    int char_count = 0;

    static int recurse_count = 0;
    int result = 0;

    if (unw_get_reg_p(&eunw->cursor, UNW_REG_IP, &reg_pc) < 0)
        reg_pc = ULONG_MAX;

    if (reg_pc && (*prev_pc == reg_pc) &&
        (recurse_count <= MAX_RECURSE_COUNT)) {
        recurse_count++;

        result = unw_step_p(&eunw->cursor);

        if (result < 0)
            snprintf(buf_cursor, buf_size,
                     "Get up one level failed. Error %d\n", result);

        return result;
    } else {
        if (recurse_count) {
            char_count = snprintf(buf_cursor,
                                  buf_size,
                                  TMPL_REPEAT,
                                  *prev_pc,
                                  recurse_count);
            buf_cursor += char_count;
            buf_size -= char_count;

            if (recurse_count >= MAX_RECURSE_COUNT) {
                exchnd_backend_store(buffer, strlen(buffer));
                exchnd_log_error("Too many recursion, aborting unwinding.\n");
                recurse_count = 0;

                return -1;
            }

            recurse_count = 0;
        }

        *prev_pc = reg_pc;
    }

    result = exchnd_get_func_at_address(reg_pc,
                                        eunw,
                                        buf_cursor,
                                        buf_size,
                                        last_func);

    exchnd_backend_store(buffer, strlen(buffer));

    result = unw_step_p(&eunw->cursor);

    if (extend == EXCHND_FB_EXTENDED)
        exchnd_extend_bt(eunw, prev_sp);

    if (result < 0)
        exchnd_log_error("Get up one level failed. Error %d\n", result);

    return result;
}

static int __get_func_from_stack(struct exchnd_unwind *eunw)
{
    unsigned int i = 0;
    char buffer[MAX_MSG_LEN];
    char *buf_cursor = buffer;

    unw_word_t pc = 0x0;
    unw_word_t sp = 0x0;
    unw_word_t top_sp = 0x0;
    int skipped = 0;

    struct exchnd_maps *sp_map = NULL;

    if (unw_get_reg_p(&eunw->cursor, get_arch_sp_reg(), &sp) < 0) {
        exchnd_log_error("Stack address not found !");
        return -1;
    }

    sp_map = get_map(sp, eunw);

    if (!sp_map) {
        exchnd_log_error("Stack map not found !");
        return -1;
    }

    if (guess_size > 0) {
        top_sp = sp + guess_size;
        exchnd_log_error("Searching in %d bytes from %p.\n", guess_size, sp);
    } else {
        top_sp = sp_map->end;
        exchnd_log_error("Searching from %p.\n", guess_size);
    }

    for (i = sp; (i < top_sp) && (i < sp_map->end); i += ULSZ) {
        struct exchnd_maps *map = NULL;
        eunw->accessors->access_mem(eunw->addr_space,
                                    i,
                                    &pc,
                                    0,
                                    eunw->upt_info);

        if (!pc) {
            skipped++;

            if (skipped == 4) {
                top_sp += 4 * ULSZ;
                skipped = 0;
            }

            continue;
        }

        skipped = 0;

        map = get_map(pc, eunw);

        if (map && *map->name && (*map->name != '[')) {
            memset(buffer, 0, MAX_MSG_LEN);

            if (!exchnd_get_func_at_address(pc,
                                            eunw,
                                            buf_cursor,
                                            MAX_MSG_LEN,
                                            NULL))
                exchnd_backend_store(buffer, strlen(buffer));
        }
    }

    exchnd_log_error("Stopped at %p.\n", top_sp);

    return 0;
}

int get_func_from_stack(int pid, struct exchnd_unwind *eunw)
{
    int result;

    if (libunwind_prepare(pid, eunw)) {
        exchnd_log_error("Unable to access remote process.");
        return -1;
    }

    result = __get_func_from_stack(eunw);

    libunwind_clean(eunw);

    return result;
}

static void exchnd_set_method(int meth)
{
#if 0 /*def unw_set_unwind_method */
    unw_set_unwind_method_p(meth);
#else
    char smeth[10] = { 0 };

    snprintf(smeth, 10, "%d", meth);
    setenv("UNW_ARM_UNWIND_METHOD", smeth, 1);
    libunwind_unload();
    libunwind_load();
#endif
}

int __log_backtrace(int pid, int extend, struct exchnd_unwind *eunw)
{
    int result;
    int max_count = max_bt_depth;
    char buffer[MAX_MSG_LEN];
    char last_func[MAX_MSG_LEN];

    unw_word_t prev_pc = 0x0;
    unw_word_t prev_sp = 0x0;

    if (extend == EXCHND_FB_FRAME)
        exchnd_set_method(EXH_METHOD_FRAME);

    if (libunwind_prepare(pid, eunw)) {
        exchnd_log_error("Unable to access remote process.");
        return -1;
    }

    if (extend == EXCHND_FB_EXTENDED) {

        snprintf(buffer, MAX_MSG_LEN, "Extracted from registers:");
        exchnd_backend_store(buffer, strlen(buffer));

        exchnd_indirect_regs_dump(eunw);

        snprintf(buffer, MAX_MSG_LEN, "---");
        exchnd_backend_store(buffer, strlen(buffer));

        snprintf(buffer, MAX_MSG_LEN, "Extracted from stack:");
        exchnd_backend_store(buffer, strlen(buffer));
    }

    /* Initialize previous SP value */
    if (unw_get_reg_p(&eunw->cursor, get_arch_sp_reg(), &prev_sp) < 0)
        prev_sp = 0UL;

    do
        result = exchnd_unwind(eunw, &prev_pc, &prev_sp, extend, last_func);
    while ((result > 0) && (max_count-- > 0));

    libunwind_clean(eunw);

    if (extend == EXCHND_FB_FRAME)
        exchnd_set_method(EXH_METHOD_ALL);

    if (result == 0)
        result = exchnd_validate_pc(eunw, prev_pc, last_func);

    /* Don't try fall backs if the backtrace looped */
    if (max_count <= 0) {
        snprintf(buffer, MAX_MSG_LEN, "Aborted backtrace, too many functions.");
        exchnd_backend_store(buffer, strlen(buffer));
        result = 0;
    }

    return result;
}

int __collector_stack_dump(ExchndInterface_t *exh_if,
                           struct exchnd_message_header *header,
                           char *data)
{
    struct exchnd_unwind eunw = { 0 };

    unw_word_t reg_sp;
    unw_word_t start_stack = 0;

    struct dump_mem dm = {
        /* Line to dump should be even */
        .l = 32,
        .w = 8
    };

#if (EXCHND_VERSION > 0x206)

    if ((action & EXCHND_EXT_STACK) && (header->sig & (1 << SIGABRT))) {
        unsigned int bytes_per_line = ULSZ * dm.w;

        dm.l = 4096 / bytes_per_line;
    }

#endif

    /* Attach only if it is not already attached */
    if (exh_if->PTRACE_attached == EXH_DETACHED) {
        int mustkill = (action & EXCHND_KILL);

        if (pt_attach(header->pid, mustkill) != 0)
            return -1;

        exh_if->PTRACE_attached = EXH_THREAD;
    }

    if (libunwind_prepare(header->pid, &eunw)) {
        exchnd_log_error("Unable to access remote process.");
        return -1;
    }

    if (unw_get_reg_p(&eunw.cursor, UNW_REG_SP, &reg_sp) < 0) {
        exchnd_log_error("Failed to get stack pointer");
        return -1;
    }

    start_stack = strtoul((char *)data, NULL, BASE_HEX);

    exchnd_log_error("mm->start_stack: 0x%x\n", start_stack);

#if (EXCHND_VERSION > 0x206)

    if ((action & EXCHND_EXT_STACK) && (header->sig & (1 << SIGABRT)))
        reg_sp -= 12 * ULSZ;

#endif

    dm.start = reg_sp;
    dm.end = start_stack;

    exchnd_dump_aligned_mem(&eunw, &dm);

    libunwind_clean(&eunw);

    return 0;
}

int __collector_mem_dump(ExchndInterface_t *exh_if,
                         struct exchnd_message_header *header,
                         unsigned long faulty_add)
{
    struct exchnd_unwind eunw = { 0 };

    struct dump_mem dm = {
        /* Line to dump should be even */
        .l = 6,
        .w = 8
    };

    /* Attach only if it is not already attached */
    if (exh_if->PTRACE_attached == EXH_DETACHED) {
        int mustkill = (action & EXCHND_KILL);

        if (pt_attach(header->pid, mustkill) != 0)
            return -1;

        exh_if->PTRACE_attached = EXH_THREAD;
    }

    if (libunwind_prepare(header->pid, &eunw)) {
        exchnd_log_error("Unable to access remote process.");
        return -1;
    }

    dm.start = faulty_add - 8 * ULSZ;
    dm.end = faulty_add + 12 * ULSZ;
    exchnd_dump_aligned_mem(&eunw, &dm);

    libunwind_clean(&eunw);

    return 0;
}